﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/imagebuilder/ImagebuilderRequest.h>
#include <aws/imagebuilder/Imagebuilder_EXPORTS.h>

#include <utility>

namespace Aws {
namespace imagebuilder {
namespace Model {

/**
 */
class RetryImageRequest : public ImagebuilderRequest {
 public:
  AWS_IMAGEBUILDER_API RetryImageRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "RetryImage"; }

  AWS_IMAGEBUILDER_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The source image Amazon Resource Name (ARN) to retry.</p>
   */
  inline const Aws::String& GetImageBuildVersionArn() const { return m_imageBuildVersionArn; }
  inline bool ImageBuildVersionArnHasBeenSet() const { return m_imageBuildVersionArnHasBeenSet; }
  template <typename ImageBuildVersionArnT = Aws::String>
  void SetImageBuildVersionArn(ImageBuildVersionArnT&& value) {
    m_imageBuildVersionArnHasBeenSet = true;
    m_imageBuildVersionArn = std::forward<ImageBuildVersionArnT>(value);
  }
  template <typename ImageBuildVersionArnT = Aws::String>
  RetryImageRequest& WithImageBuildVersionArn(ImageBuildVersionArnT&& value) {
    SetImageBuildVersionArn(std::forward<ImageBuildVersionArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Unique, case-sensitive identifier you provide to ensure idempotency of the
   * request. For more information, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
   * idempotency</a> in the <i>Amazon EC2 API Reference</i>.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  RetryImageRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_imageBuildVersionArn;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_imageBuildVersionArnHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace imagebuilder
}  // namespace Aws
