﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/mq/MQRequest.h>
#include <aws/mq/MQ_EXPORTS.h>
#include <aws/mq/model/AuthenticationStrategy.h>
#include <aws/mq/model/EngineType.h>

#include <utility>

namespace Aws {
namespace MQ {
namespace Model {

/**
 * <p>Creates a new configuration for the specified configuration name. Amazon MQ
 * uses the default configuration (the engine type and version).</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/CreateConfigurationRequest">AWS
 * API Reference</a></p>
 */
class CreateConfigurationRequest : public MQRequest {
 public:
  AWS_MQ_API CreateConfigurationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateConfiguration"; }

  AWS_MQ_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Optional. The authentication strategy associated with the configuration. The
   * default is SIMPLE.</p>
   */
  inline AuthenticationStrategy GetAuthenticationStrategy() const { return m_authenticationStrategy; }
  inline bool AuthenticationStrategyHasBeenSet() const { return m_authenticationStrategyHasBeenSet; }
  inline void SetAuthenticationStrategy(AuthenticationStrategy value) {
    m_authenticationStrategyHasBeenSet = true;
    m_authenticationStrategy = value;
  }
  inline CreateConfigurationRequest& WithAuthenticationStrategy(AuthenticationStrategy value) {
    SetAuthenticationStrategy(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Required. The type of broker engine. Currently, Amazon MQ supports ACTIVEMQ
   * and RABBITMQ.</p>
   */
  inline EngineType GetEngineType() const { return m_engineType; }
  inline bool EngineTypeHasBeenSet() const { return m_engineTypeHasBeenSet; }
  inline void SetEngineType(EngineType value) {
    m_engineTypeHasBeenSet = true;
    m_engineType = value;
  }
  inline CreateConfigurationRequest& WithEngineType(EngineType value) {
    SetEngineType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The broker engine version. Defaults to the latest available version for the
   * specified broker engine type. For more information, see the <a
   * href="https://docs.aws.amazon.com//amazon-mq/latest/developer-guide/activemq-version-management.html">ActiveMQ
   * version management</a> and the <a
   * href="https://docs.aws.amazon.com//amazon-mq/latest/developer-guide/rabbitmq-version-management.html">RabbitMQ
   * version management</a> sections in the Amazon MQ Developer Guide.</p>
   */
  inline const Aws::String& GetEngineVersion() const { return m_engineVersion; }
  inline bool EngineVersionHasBeenSet() const { return m_engineVersionHasBeenSet; }
  template <typename EngineVersionT = Aws::String>
  void SetEngineVersion(EngineVersionT&& value) {
    m_engineVersionHasBeenSet = true;
    m_engineVersion = std::forward<EngineVersionT>(value);
  }
  template <typename EngineVersionT = Aws::String>
  CreateConfigurationRequest& WithEngineVersion(EngineVersionT&& value) {
    SetEngineVersion(std::forward<EngineVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Required. The name of the configuration. This value can contain only
   * alphanumeric characters, dashes, periods, underscores, and tildes (- . _ ~).
   * This value must be 1-150 characters long.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateConfigurationRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Create tags when creating the configuration.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateConfigurationRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateConfigurationRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  AuthenticationStrategy m_authenticationStrategy{AuthenticationStrategy::NOT_SET};
  bool m_authenticationStrategyHasBeenSet = false;

  EngineType m_engineType{EngineType::NOT_SET};
  bool m_engineTypeHasBeenSet = false;

  Aws::String m_engineVersion;
  bool m_engineVersionHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace MQ
}  // namespace Aws
