from typing import Optional, List

from pydantic import BaseModel
from sqlalchemy import Column, Integer, ForeignKey, String, Double, Boolean
from sqlalchemy.orm import relationship

from ..base import Base
from .region import RegionShortInDB, Region
from .frame_times import FrameTimes, FrameTimesInDB, FrameTimesCreate


class HotspotAnalysisCustomField(Base):
    __tablename__ = 'hotspot_analysis_custom_fields'
    id = Column(Integer, primary_key=True, index=True)
    name = Column(String)
    value = Column(String)

    hotspot_analysis_id = Column(Integer, ForeignKey('hotspot_analysis.id'))
    hotspot_analysis = relationship('HotspotAnalysis', back_populates='custom_fields')


class HotspotAnalysisCustomFieldInDB(BaseModel):
    name: str
    value: str

    class Config:
        from_attributes = True


class HotspotAnalysisCustomFieldCreate(BaseModel):
    name: str
    value: str

    class Config:
        from_attributes = True


class HotspotAnalysisCustomChartRow(Base):
    __tablename__ = 'hotspot_analysis_custom_chart_rows'
    id = Column(Integer, primary_key=True, index=True)
    row_index = Column(Integer)
    value = Column(Double)

    hotspot_analysis_id = Column(Integer, ForeignKey('hotspot_analysis.id'))
    hotspot_analysis = relationship('HotspotAnalysis', back_populates='custom_chart_rows')


class HotspotAnalysisCustomChartRowInDB(BaseModel):
    row_index: int
    value: float

    class Config:
        from_attributes = True


class HotspotAnalysisCustomChartRowCreate(BaseModel):
    row_index: int
    value: float

    class Config:
        from_attributes = True


class HotspotAnalysis(Base):
    __tablename__ = 'hotspot_analysis'
    id = Column(Integer, primary_key=True, index=True)
    # Capture Info
    process_name = Column(String, default='')
    process_id = Column(Integer, nullable=True)
    start_time_s = Column(Double, nullable=True)
    end_time_s = Column(Double, nullable=True)
    duration_s = Column(Double, nullable=True)
    # Region Info
    region_description = Column(String, default='')
    number_of_regions = Column(Integer, nullable=True)
    fastest_ms = Column(Double, nullable=True)
    slowest_ms = Column(Double, nullable=True)
    median_ms = Column(Double, nullable=True)
    mean_ms = Column(Double, nullable=True)
    average_micro_stutters_ms = Column(Double, nullable=True)
    region_count = Column(Integer, nullable=True)
    # General Info
    app_health_gpu_bound = Column(Boolean, default=False)
    hotspot_range = Column(String)
    hotspot_report_threading = Column(String)

    report_id = Column(Integer, ForeignKey('report.id'))
    report = relationship('Report', back_populates='hotspot_analysis')
    regions = relationship(Region, back_populates='hotspot_analysis')
    frame_times = relationship(FrameTimes, back_populates="hotspot_analysis")
    custom_fields = relationship(HotspotAnalysisCustomField, back_populates='hotspot_analysis')
    custom_chart_rows = relationship(HotspotAnalysisCustomChartRow, back_populates='hotspot_analysis')
    custom_chart_title = Column(String, nullable=True)
    custom_chart_x_axis_name = Column(String, nullable=True)
    custom_chart_y_axis_name = Column(String, nullable=True)


class HotspotAnalysisShortInDB(BaseModel):
    id: int
    hotspot_range: str
    hotspot_report_threading: str

    class Config:
        from_attributes = True


class HotspotAnalysisInDB(HotspotAnalysisShortInDB):
    process_name: str
    process_id: Optional[int]
    start_time_s: Optional[float]
    end_time_s: Optional[float]
    duration_s: Optional[float]

    region_description: str
    number_of_regions: Optional[int]
    fastest_ms: Optional[float]
    slowest_ms: Optional[float]
    median_ms: Optional[float]
    mean_ms: Optional[float]
    average_micro_stutters_ms: Optional[float]
    region_count: Optional[int]

    app_health_gpu_bound: bool

    regions: List[RegionShortInDB]
    frame_times: List[FrameTimesInDB]
    custom_fields: List[HotspotAnalysisCustomFieldInDB]
    custom_chart_rows: List[HotspotAnalysisCustomChartRowInDB]
    custom_chart_title: Optional[str]
    custom_chart_x_axis_name: Optional[str]
    custom_chart_y_axis_name: Optional[str]

    class Config:
        from_attributes = True


class HotspotAnalysisBaseCreate(BaseModel):
    report_id: int

    process_name: str
    process_id: Optional[int]
    start_time_s: Optional[float]
    end_time_s: Optional[float]
    duration_s: Optional[float]

    region_description: Optional[str]
    number_of_regions: Optional[int]
    fastest_ms: Optional[float]
    slowest_ms: Optional[float]
    median_ms: Optional[float]
    mean_ms: Optional[float]
    average_micro_stutters_ms: Optional[float]
    region_count: Optional[int]

    app_health_gpu_bound: bool
    hotspot_range: str
    hotspot_report_threading: str
    custom_chart_title: Optional[str] = None
    custom_chart_x_axis_name: Optional[str] = None
    custom_chart_y_axis_name: Optional[str] = None

    class Config:
        from_attributes = True


class HotspotAnalysisCreate(HotspotAnalysisBaseCreate):

    frame_times: List[FrameTimesCreate] = []
    custom_fields: List[HotspotAnalysisCustomFieldCreate] = []
    custom_chart_rows: List[HotspotAnalysisCustomChartRowCreate] = []

    class Config:
        from_attributes = True
